<?php
namespace NeoRenameBeta\NeoGlobal;




function register_rest_endpoint($path, $method, $permission, $callback) {
    if (!str_starts_with($path, "/wp-json/neo/")) { throw new GlobalException("The endpoint path must start with '/wp-json/neo/'."); }
    if (!in_array($permission, ["admin", "editor", "public"])) { throw new GlobalException("The permission must be 'admin', 'editor' or 'public'."); }
    \NeoRenameBeta\NeoGlobal\add_action_hook("rest_api_init", function () use ($path, $method, $permission, $callback) {
        foreach (is_array($method) ? $method : [$method] as $http_method) {
            if (!in_array($http_method, ["GET", "POST"])) { throw new GlobalException("Invalid method $http_method - it must be 'GET' or 'POST'. Don't use e.g. PUT because the request body isn't passed through e.g. in the WP Playground."); }
            $path_parts = explode("/", $path);
            $namespace = $path_parts[2]; $route = "/" . implode("/", array_slice($path_parts, 3));
            register_rest_route($namespace, $route, [
                "methods" => $http_method,
                "permission_callback" => function ($request) use ($permission) {

                    if ($permission === "public") { return true; }
                    
                    $nonce = $request->get_header("X-WP-Nonce") ?: $request->get_param("_wpnonce");
                    if (!wp_verify_nonce($nonce, "wp_rest")) { return new \WP_Error("invalid_nonce", "Invalid nonce", ["status" => 401]); }
                    
                    if (!is_user_logged_in()) { return new \WP_Error("not_logged_in", "You are not currently logged in.", ["status" => 401]); }
                    if ($permission === "admin"  && !current_user_can("manage_options"))    { return new \WP_Error("permission_denied", "Your user role does not have permission to access this endpoint.", ["status" => 403]); }
                    if ($permission === "editor" && !current_user_can("edit_others_posts")) { return new \WP_Error("permission_denied", "Your user role does not have permission to access this endpoint.", ["status" => 403]); }

                    return true;
                },
                "callback" => function ($request) use ($callback) {
                    $get_param = function ($key) use ($request) {
                        if (sanitize_key($key) !== $key) { throw new \NeoRenameBeta\NeoGlobal\GlobalException("WP only allows alphanumeric characters and underscores for the parameter name!"); }
                        return $request->get_param($key) ?? $request->get_json_params()[$key] ?? null;
                    };
                    try {
                        if (count((new \ReflectionFunction($callback))->getParameters()) === 1) {
                            return $callback($get_param);
                        } else {
                            return $callback();
                        }
                    } catch (\Exception $e) {
                        if (WP_DEBUG) {
                            header("HTTP/1.1 500 Internal Server Error");
                            echo $e->getMessage() . "\n\n" . $e->getTraceAsString();
                            exit;
                        }
                        throw $e;
                    }
                },
            ]);
        }
    });
}


function get_rest_endpoint_url($path) {
    if (!did_action("init")) { throw new GlobalException("WP is not initialized yet. Cannot use the WP function for rest_url(). Please call this function after the 'init' action."); }
    if (!str_starts_with($path, "/wp-json/")) { throw new GlobalException("The endpoint path must start with '/wp-json/'."); }
    $path = str_replace("/wp-json/", "", $path);
    return rest_url($path);
}
