(() => { 
    class NeoSwitch extends HTMLElement {
        constructor() {
            super();
            
            this.attachShadow({ mode: "open" });
            this.shadowRoot.innerHTML = `
        <style>
          /* Komponente nimmt volle Breite ein */
          :host {
            display: flex;
            width: 100%;
          }
          /* Deaktivierter Zustand: Schalter grau und keine Interaktion */
          :host([disabled]) .switch-label { pointer-events: none; } /* Keine Interaktion */
          :host([disabled]) .switch-label .switch-slider:not(.more-specificity) { background-color: #d0d0d0; } /* Schalter grau. :not -> Grünen Hintergrund bei angeschaltetem Schalter überschreiben */
          /* Container für Labeltext und Schalter, horizontal verteilt */
          .outer-container {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 20px;
            width: 100%;
          }
          /* Stil für den optionalen Text links */
          .label-text {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 1rem;
            color: #000;
          }
          /* Label als Schalter mit Accessibility-Unterstützung */
          .switch-label {
            position: relative;
            display: inline-block;
            width: 60px;
            height: 34px;
            cursor: pointer;
          }
          /* Verstecktes Checkbox-Input */
          .switch-label input {
            opacity: 0;
            width: 0;
            height: 0;
          }
          /* Basis-Stil des Schiebereglers */
          .switch-slider {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #D000D0; /* Farbe für "aus" */
            transition: background-color 0.2s;
            border-radius: 34px;
          }
          /* Kreis im Schalter */
          .switch-circle {
            position: absolute;
            height: 26px;
            width: 26px;
            left: 4px;
            bottom: 4px;
            background-color: white;
            transition: left 0.2s;
            border-radius: 50%;
          }
          /* Wenn aktiviert und nicht im Ladezustand */
          .switch-label input:checked + .switch-slider:not(.loading) {
            background-color: #00D000;
          }
          .switch-label input:checked + .switch-slider:not(.loading) .switch-circle {
            left: calc(100% - 26px - 4px);
          }
          /* Im Ladezustand: Kreis auch nach rechts, wenn aktiviert */
          .switch-label input:checked + .switch-slider.loading .switch-circle {
            left: calc(100% - 26px - 4px);
          }
          /* Fokus-Stil */
          .switch-label input:focus + .switch-slider {
            box-shadow: 0 0 1px #00D000;
          }
          /* Lade-Zustand: Gelber Hintergrund und Rotation im Kreis */
          .switch-slider.loading {
            background-color: #FFC003;
          }
          .switch-slider.loading .switch-circle {
            animation: switch-loading 1s infinite linear;
            border: 8px solid #606060;
            border-top-color: transparent;
            border-bottom-color: transparent;
            box-sizing: border-box;
          }
          @keyframes switch-loading {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
          }
        </style>
        <div class="outer-container">
          <!-- Optionaler Text (Label) -->
          <span class="label-text"></span>
          <!-- Pro-Krone (wird bei Bedarf dynamisch eingefügt) -->
          <!-- Schalter-Label mit ARIA-Attributen -->
          <label class="switch-label" role="switch" tabindex="0" aria-checked="false" aria-disabled="false">
            <input type="checkbox" name="neo-switch-input" aria-label="Toggle switch">
            <span class="switch-slider">
              <span class="switch-circle"></span>
            </span>
          </label>
        </div>
      `;

            
            this._loadingStart = 0;
            this._loadingTimeout = null;
            this._suppressLoadingAttrChange = false;

        
            this._proCheckInFlight = false;      
         

         
            this._proDisabledTitle = null;       
            this._proDisabledTitleInFlight = false;
           
        }

        
        static get observedAttributes() {
            return ["loading", "checked", "label", "disabled", "disabled-title", "only-enabled-if-pro"];
        }

        attributeChangedCallback(name, oldValue, newValue) {
            if (name === "loading") {
                if (this._suppressLoadingAttrChange) {
                    this._suppressLoadingAttrChange = false;
                    this._updateLoading();
                    this._updateAccessibility();
                    return;
                }
                if (newValue !== null) {
                    
                    this._loadingStart = Date.now();
                    if (this._loadingTimeout) {
                        clearTimeout(this._loadingTimeout);
                        this._loadingTimeout = null;
                    }
                } else {
                    
                    const elapsed = Date.now() - (this._loadingStart || 0);
                    const minDuration = 500;
                    if (elapsed < minDuration) {
                        const remaining = minDuration - elapsed;
                        this._suppressLoadingAttrChange = true;
                        this.setAttribute("loading", "true");
                        this._loadingTimeout = setTimeout(() => {
                            this._suppressLoadingAttrChange = true;
                            this.removeAttribute("loading");
                            this._suppressLoadingAttrChange = false;
                            this._loadingTimeout = null;
                        }, remaining);
                        return;
                    }
                }
                this._updateLoading();
                this._updateAccessibility();
            }
            if (name === "checked") {
                
                const isChecked = newValue !== null;
                this.shadowRoot.querySelector("input").checked = isChecked;
                this._updateAccessibility();
            }
            if (name === "label") {
                
                this._updateLabel();
            }
            if (name === "disabled") {
                
                this._updateAccessibility();
                this._updateDisabledTitle();
            }
            if (name === "disabled-title") {
                
                this._updateDisabledTitle();
            }
            if (name === "only-enabled-if-pro") {
                
                this._applyProState();
            }
        }

        connectedCallback() {
            
            this._updateLoading();
            this._updateAccessibility();
            this._updateLabel();
            this._updateDisabledTitle();
            
            
            this.shadowRoot.querySelector("input").checked = this.hasAttribute("checked");

            const labelElem = this.shadowRoot.querySelector(".switch-label");

            





            labelElem.addEventListener("click", (e) => {
                
                if (this.loading || this.disabled) {
                    e.preventDefault();
                    return;
                }
                e.preventDefault(); 
                this._handleToggle();
            });

            
            labelElem.addEventListener("keydown", (e) => {
                if ((e.key === " " || e.key === "Enter") && !this.loading && !this.disabled) {
                    e.preventDefault();
                    this._handleToggle();
                }
            });

   
            this._applyProState();
        }

        
        get checked() {
            return this.shadowRoot.querySelector("input").checked;
        }
        set checked(val) {
            this.shadowRoot.querySelector("input").checked = val;
            if (val) {
                this.setAttribute("checked", "");
            } else {
                this.removeAttribute("checked");
            }
            this._updateAccessibility();
        }

        
        get loading() {
            return this.hasAttribute("loading");
        }
        set loading(val) {
            if (val) {
                this.setAttribute("loading", "true");
            } else {
                this.removeAttribute("loading");
            }
        }

        
        get disabled() {
            return this.hasAttribute("disabled");
        }
        set disabled(val) {
            if (val) {
                this.setAttribute("disabled", "true");
            } else {
                this.removeAttribute("disabled");
            }
        }

        
        _updateLoading() {
            const slider = this.shadowRoot.querySelector(".switch-slider");
            if (this.loading) {
                slider.classList.add("loading");
            } else {
                slider.classList.remove("loading");
            }
        }

        
        _updateAccessibility() {
            const labelElem = this.shadowRoot.querySelector(".switch-label");
            labelElem.setAttribute("aria-checked", this.checked ? "true" : "false");
            
            labelElem.setAttribute("aria-disabled", (this.loading || this.disabled) ? "true" : "false");
        }

        
        _updateLabel() {
            const labelTextElem = this.shadowRoot.querySelector(".label-text");
            if (this.hasAttribute("label") && this.getAttribute("label").trim() !== "") {
                labelTextElem.textContent = this.getAttribute("label");
                labelTextElem.style.display = "flex";
            } else {
                labelTextElem.textContent = "";
                labelTextElem.style.display = "none";
            }
        }

        


        _updateDisabledTitle() {
            const labelElem = this.shadowRoot.querySelector(".switch-label");

            this.setAttribute("title", this.getAttribute("data-title-original") || "");

            
            if (this.disabled && this.hasAttribute("data-pro-disabled")) {
                this.setAttribute("data-title-original", this.getAttribute("title") || "");
                if (this._proDisabledTitle) {
                    this.setAttribute("title", this._proDisabledTitle);
                } else if (!this._proDisabledTitleInFlight) {
                    this._proDisabledTitleInFlight = true;
                    (async () => {
                        const { neo__ } = await import("./_global-translation.js");
                        this._proDisabledTitle = neo__("Get neoPro now to unlock this feature", "Jetzt neoPro holen, um dieses Feature freizuschalten");
                        this.setAttribute("title", this._proDisabledTitle);
                        this._proDisabledTitleInFlight = false;
                    })().catch(() => {
                        
                        this.setAttribute("title", "Get neoPro");
                        this._proDisabledTitleInFlight = false;
                    });
                }
                return;
            }

            
            if (this.disabled && this.hasAttribute("disabled-title")) {
                this.setAttribute("data-title-original", this.getAttribute("title") || "");
                this.setAttribute("title", this.getAttribute("disabled-title")); 
            }
        }

        








        async _handleToggle() {
            
            if (this.disabled || this.loading) return;

            
            const previousState = this.checked;
            
            this.checked = !previousState;

            let callbackResult;
            try {
                
                if (typeof this.onchange === "function") {
                    callbackResult = this.onchange({ checked: this.checked });
                    
                    if (callbackResult && typeof callbackResult.then === "function") {
                        
                        this.loading = true;
                        
                        const minLoadingTime = new Promise(resolve => setTimeout(resolve, 500));
                        
                        await Promise.all([callbackResult, minLoadingTime]);
                    }
                }
            } catch (error) {
                
                this.checked = previousState;
                
                throw error;
            } finally {
                
                this.loading = false;
            }
        }

        
        toggle() { return this._handleToggle(); }


        


        _applyProState() {
            const hasProAttr = this.hasAttribute("only-enabled-if-pro");
            const outerContainer = this.shadowRoot.querySelector(".outer-container");
            const labelTextElem = outerContainer.querySelector(".label-text");
            
            let crownElem = outerContainer.querySelector("neo-pro-crown-neo-rename_beta");
            if (hasProAttr) {
                if (!crownElem) {
                    crownElem = document.createElement("neo-pro-crown-neo-rename_beta");
                    labelTextElem.appendChild(crownElem);
                }
                
                this._setProDisabled(true);

                 
                if (!this._proCheckInFlight) {
                    this._proCheckInFlight = true;
                    (async () => { 
                        try {
                            
                            

                            


                            this._setProDisabled(true);
                        
                        } catch (error) {
                            
                            try {
                                const { neoError } = await import("./_global-log.js");
                                neoError(error);
                            } catch (logError) {
                                console.error("Fehler beim Laden von neoError:", logError); 
                            }
                            this._setProDisabled(true);
                        } finally {
                            this._proCheckInFlight = false;
                        }
                    })(); 
                }
            } else {
               
                if (crownElem) crownElem.remove();
                this._setProDisabled(false);
            }
        }

        





        _setProDisabled(shouldDisable) {
            if (shouldDisable) {
                if (!this.hasAttribute("disabled")) {
                    this.setAttribute("disabled", "true");
                    this.setAttribute("data-pro-disabled", "true");
                    this._updateAccessibility();
                    this._updateDisabledTitle();
                } else if (!this.hasAttribute("data-pro-disabled")) {
                    
                }
            } else {
                if (this.hasAttribute("data-pro-disabled")) {
                    this.removeAttribute("data-pro-disabled");
                    this.removeAttribute("disabled");
                    this._updateAccessibility();
                    this._updateDisabledTitle();
                }
            }
        }
    }

    
    if (!document.currentScript.hasAttribute("data-web-component-name")) {
        throw new Error("You have to provide a 'data-web-component-name' attribute for the script tag!");
    }
    
    customElements.define(document.currentScript.getAttribute("data-web-component-name"), NeoSwitch);
})(); 
