<?php
namespace NeoRenameBeta\NeoLog;









\NeoRenameBeta\NeoGlobal\register_backend_page("neo-log", function () {
    
    $log_dir = \NeoRenameBeta\NeoGlobal\cache_path("neo-log");
    $warnings = [];
    $file_infos = [];
    $all_logs = [];
    $per_file_limit = isset($_GET["limit"]) ? ((int) $_GET["limit"]) : 100;
    
    if (!is_dir($log_dir)) { echo "Warning: The log directory does not exist: {$log_dir}"; exit; }
    $log_files = \NeoRenameBeta\NeoGlobal\iterate_all_files($log_dir);
    if (empty($log_files)) { echo "No log files found in {$log_dir}"; exit; }
    
    foreach ($log_files as $full_path) {
        
        $file_size_bytes = filesize($full_path);
        if ($file_size_bytes === false) { $file_size_bytes = 0; $warnings[] = "Warning: Could not get file size for {$full_path}"; }
        $file_size_kb = round($file_size_bytes / 1024);
        
        $file_logs = [];
        $handle = @fopen($full_path, "r");
        if (!$handle) { $warnings[] = "Warning: Could not open file {$full_path} for reading"; $file_infos[] = ["file"  => basename($full_path), "count" => 0, "size"  => $file_size_kb ]; continue; }
        
        $line_counter = 0;
        while (($line = fgets($handle)) !== false) {
            $line_counter++; if ($line_counter > $per_file_limit) { break; }
            $line = trim($line); if ($line === "") { continue; }
            try { $decoded = \NeoRenameBeta\NeoGlobal\json_decode_better($line); } catch (\Throwable $error) { $warnings[] = "Warning: Could not decode JSON line in file $full_path: $line - $error"; continue; }
            $decoded["_raw"] = $line;
            if (!isset($decoded["date"])) { $warnings[] = "Warning: Missing 'date' in file {$full_path}: {$line}"; continue; }
            
            if (!isset($decoded["message"])) {
                $json_string = \NeoRenameBeta\NeoGlobal\json_encode_better($decoded);
                if (strlen($json_string) > 1000) { $json_string = substr($json_string, 0, 1000) . "..."; }
                $decoded["message"] = $json_string;
            }

            $file_logs[] = ["date" => $decoded["date"], "message" => $decoded["message"], "raw" => $decoded["_raw"]];
        }
        fclose($handle);
        
        $common_prefix = function ($list_of_word_arrays) {
            if (empty($list_of_word_arrays)) { return []; }
            $common_prefix_words = [];
            $smallest_number_of_elements = min(array_map("count", $list_of_word_arrays));
            for ($i = 0; $i < $smallest_number_of_elements; $i++) {
                $all_the_same = count(\NeoRenameBeta\NeoGlobal\array_unique_better(array_column($list_of_word_arrays, $i))) === 1;
                if ($all_the_same) { $common_prefix_words[] = $list_of_word_arrays[0][$i]; } else { break; }
            }
            return $common_prefix_words;
        };

        $common_suffix = function ($list_of_word_arrays) use ($common_prefix) {
            return array_reverse($common_prefix(array_map("array_reverse", $list_of_word_arrays)));
        };

        $deduplicated = [];
        foreach ($file_logs as $log_entry) {
            $msg_words = explode(" ", $log_entry["message"]);
            $matched_group = false;
            
            foreach ($deduplicated as &$group) {
                $starts_with_same = (empty($msg_words) || empty($group["prefix"])) ? false : str_starts_with($msg_words[0], $group["prefix"][0]);
                $ends_with_same   = (empty($msg_words) || empty($group["suffix"])) ? false : str_ends_with(end($msg_words), end($group["suffix"]));
                if ($starts_with_same || $ends_with_same) {
                    $all_msgs = $group["middles"];
                    $all_msgs[] = $msg_words;
                    $new_prefix = $common_prefix($all_msgs);
                    $new_suffix = $common_suffix($all_msgs);
                    if (empty($new_prefix) && empty($new_suffix)) { continue; }
                    $group["prefix"] = $new_prefix; $group["suffix"] = $new_suffix; $group["middles"] = $all_msgs; $group["dates"][] = $log_entry["date"]; $group["count"]++; $group["raw_lines"][] = $log_entry["raw"];
                    $matched_group = true; break;
                }
            }
            unset($group);
            
            if (!$matched_group) { $deduplicated[] = ["prefix" => $msg_words, "suffix" => $msg_words, "middles" => [$msg_words], "dates" => [$log_entry["date"]], "count" => 1, "raw_lines" => [ $log_entry["raw"] ]]; }
        }


        $final_logs_this_file = [];
        foreach ($deduplicated as $group) {
            
            usort($group["dates"], function($a, $b) { return strcmp($b, $a); });
            $date_newest = $group["dates"][0];
            $date_oldest = end($group["dates"]);
            $duration = \NeoRenameBeta\NeoGlobal\timestamp_from_utc_date_string($date_newest) - \NeoRenameBeta\NeoGlobal\timestamp_from_utc_date_string($date_oldest);
            $new_prefix = $common_prefix($group["middles"]);
            $new_suffix = $common_suffix($group["middles"]);
            
            $middle_texts = [];
            foreach ($group["middles"] as $oneMsgWords) {
                $prefix_len = count($new_prefix);
                $suffix_len = count($new_suffix);
                $middle_len = count($oneMsgWords) - $prefix_len - $suffix_len;
                if ($middle_len < 0) { $middle_len = 0; }
                $middle = array_slice($oneMsgWords, $prefix_len, $middle_len);
                $middle_texts[] = implode(" ", $middle);
            }

            $middle_texts = \NeoRenameBeta\NeoGlobal\array_unique_better($middle_texts);
            if ($group["count"] === 1) {
                $final_message = implode(" ", $group["middles"][0]);
            } else {
                
                $prefix_str = implode(" ", $new_prefix);
                $suffix_str = implode(" ", $new_suffix);
                $curly_content_plain = implode(", ", $middle_texts);
                
                $curly_content = '{' . (mb_strlen($curly_content_plain) > 100 ? mb_substr($curly_content_plain, 0, 100) . "..." : $curly_content_plain) . '}';

                $final_message = trim($prefix_str . " " . $curly_content . " " . $suffix_str);
            }

            $final_logs_this_file[] = ["date_newest" => $date_newest, "duration" => $duration, "count" => $group["count"], "file" => basename($full_path), "message" => $final_message, "raw" => implode("\n", $group["raw_lines"])];
        }

        $file_infos[] = ["file" => basename($full_path), "count" => array_sum(array_column($final_logs_this_file, "count")), "size" => $file_size_kb];
        
        usort($file_infos, function($a, $b) { return strcmp($a["file"], $b["file"]); });
        
        $all_logs = array_merge($all_logs, $final_logs_this_file);
    }

    usort($all_logs, function($a, $b) { return strcmp($b["date_newest"], $a["date_newest"]); });
    
    ?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8" />
    <title>neoLog - Overview</title>
    <style>

        body { font-family: Arial, sans-serif; }


        #logFileTable, #logEntriesTable { border-collapse: collapse; margin-bottom: 20px; }
        :is(#logFileTable, #logEntriesTable) :is(th, td) { padding: 5px 8px; white-space: nowrap; border: none; }
        :is(#logFileTable, #logEntriesTable) th { text-align: left; background-color: #005f99; color: white; }
        

        :is(#logFileTable, #logEntriesTable) tbody tr:nth-child(odd) { background-color: #e6f7ff; }
        :is(#logFileTable, #logEntriesTable) tbody tr:nth-child(even) { background-color: #ffffff; }


        :is(#logFileTable, #logEntriesTable) :is(th, td).right-aligned { text-align: right; }


        #logFileTable { width: 600px; }
        .warn-file, .warn-message { color: #c4a000; }
        .error-file, .error-message { color: #ff0000; }
        #logSearch { width: 300px; padding: 5px; margin-bottom: 10px; }
    </style>


</head>
<body>

<?php

    if (!empty($warnings)) {
        foreach ($warnings as $w) { echo "<p style=\"color:#c4a000;\">{$w}</p>"; }
        echo "<br>";
    }
?>

<h1>neoLog Overview</h1>
<p>Currently reading max <?php echo $per_file_limit; ?> lines per file (<a href="<?php echo esc_url(add_query_arg('limit', 1000)); ?>">change to 1000</a>).</p>


<table id="logFileTable">
    <thead><tr><th>File</th><th class="right-aligned">Count</th><th class="right-aligned">Size</th></tr></thead>
    <tbody> 
    <?php foreach ($file_infos as $fi) {
        $file_name = $fi["file"];
        $file_size = $fi["size"];
        $file_class = str_contains(strtolower($file_name), "warn") ? "warn-file" : (str_contains(strtolower($file_name), "error") ? "error-file" : "");
        $count_display = $fi["count"] . ($fi["count"] >= $per_file_limit ? " (limit)" : "");
        ?>
        <tr>
            <td class="<?php echo $file_class; ?>"><?php echo htmlspecialchars($file_name); ?></td>
            <td class="right-aligned"><?php echo $count_display; ?></td>
            <td class="right-aligned"><?php echo $file_size; ?> KB</td>
        </tr>
    <?php } ?>
    </tbody>
</table>


<input type="text" id="logSearch" placeholder="Search ..." />


<table id="logEntriesTable">
    <thead><tr><th>Date</th><th>Duration</th><th>Count</th><th>File</th><th>Message</th></tr></thead>
    <tbody>
    <?php foreach ($all_logs as $log) {
        $f_class = str_contains(strtolower($log["file"]),    "warn") ? "warn-file"    : (str_contains(strtolower($log["file"]),    "error") ? "error-file"    : "");
        $m_class = str_contains(strtolower($log["message"]), "warn") ? "warn-message" : (str_contains(strtolower($log["message"]), "error") ? "error-message" : "");
        ?>
        <tr>
            <td class="right-aligned"><?php echo htmlspecialchars($log["date_newest"]); ?></td>
            <td class="right-aligned"><?php 
                $days    = floor($log["duration"] / 86400); $remaining = $log["duration"] % 86400;
                $hours   = floor($remaining / 3600); $remaining = $remaining % 3600;
                $minutes = floor($remaining / 60);
                $duration_str = "";
                if ($days > 0)    { $duration_str .= $days . "d "; }
                if ($hours > 0)   { $duration_str .= $hours . "h "; }
                if ($minutes > 0) { $duration_str .= $minutes . "m"; }
                echo trim($duration_str) ?: "0m";
            ?></td>
            <td class="right-aligned"><?php echo $log["count"]; ?></td>
            <td class="<?php echo $f_class; ?>"><?php echo htmlspecialchars($log["file"]); ?></td>
            <td class="<?php echo $m_class; ?>">
                <span onclick="let icon = this; navigator.clipboard.writeText(this.getAttribute('data-raw')).then(() => icon.textContent = '✅').then(() => setTimeout(() => icon.textContent = '📋', 2000)).catch(err => console.error('Failed to copy log:', err));"
                      data-raw="<?php echo htmlspecialchars($log['raw'], ENT_QUOTES, 'UTF-8'); ?>"
                      style="cursor: pointer; user-select: none; margin-right: 12px;">📋</span><?php
                echo $log["message"]; 
            ?></td>
        </tr>
    <?php } ?>
    </tbody>
</table>


<script>
(() => {
    let input_field = document.getElementById("logSearch"); 
    let table_body = document.getElementById("logEntriesTable").getElementsByTagName("tbody")[0];

    
    input_field.addEventListener("keyup", () => { 
        let filter = input_field.value.toLowerCase(); 
        let rows = table_body.getElementsByTagName("tr"); 
        for (let i = 0; i < rows.length; i++) { 
            let cells = rows[i].getElementsByTagName("td"); 
            let rowMatch = false; 
            for (let j = 0; j < cells.length; j++) { 
                if (cells[j].textContent.toLowerCase().includes(filter)) { 
                    rowMatch = true; 
                    break; 
                }
            }
            rows[i].style.display = rowMatch ? "" : "none"; 
        }
    });
})();


function toggleDetail(elem) { 
    const hiddenSpan = elem.querySelector(".dedup-full"); 
    if (hiddenSpan.style.display === "none") { hiddenSpan.style.display = "inline"; } else { hiddenSpan.style.display = "none"; } 
}
</script>

</body>
</html>
<?php
});
