import { neoError } from "./_global--log.js";
import { observeOnce } from "./_global--observer.js";
import { addCacheBust, addQueryParam, addQueryParams, fitProtocolToFetchImgUrl } from "./_global--url-helper.js";
import { reloadPage } from "./_global-reload-curtain.js";
import { extractJson } from "./_global--extract-json.js";
import { neo__ } from "./_global-translation.js";
import { rename } from "./neo-rename.js";

const dialogCode = `
<div class="neo-rename-dialog">
    <div class="neo-rename-dialog-backdrop"></div>
    <div class="neo-rename-dialog-box">
        <div class="neo-rename-dialog-box-content">
            <div class="neo-rename-dialog-header-container">
                <img class="image-preview" src="" alt="${neo__("Image Preview", "Vorschau")}">
                <div class="neo-rename-dialog-input-row">
                    <img src="${neoRenameData.pluginUrl}/img/_global-neo-mini-icon.svg" alt="">
                    <input type="text" class="title-input" placeholder="${neo__("Enter image title...", "Bildtitel eingeben...")}" value="...">
                    <button class="rename-button">
                        <span class="neo-rename-button-text">${neo__("Rename", "Umbenennen")}</span>
                        <span class="neo-rename-spinner"></span>
                    </button>
                </div>
            </div>
            <div class="neo-rename-dialog-options-container neo-hidden">
                <div class="neo-rename-dialog-options">
                    <div class="neo-rename-dialog-option-table">
                        <label class="neo-rename-dialog-option">
                            <div></div>
                            <div></div>
                            <b>${neo__("Old", "Alt")}</b>
                            <b>${neo__("New", "Neu")}</b>
                        </label>

                        <label class="neo-rename-dialog-option">
                            <input type="checkbox" checked id="neo-title-checkbox">
                            <b class="neo-option-name">${neo__("Title", "Titel")}</b>
                            <span class="neo-label-old neo-title-label-old"></span>
                            <span class="neo-label-new neo-title-label"></span>
                        </label>

                        <label class="neo-rename-dialog-option">
                            <input type="checkbox" id="neo-filename-checkbox"> <!-- Do not check by default because file renaming is somewhat dangerous -->
                            <b class="neo-option-name">${neo__("Filename", "Dateiname")}</b>
                            <span class="neo-label-old neo-filename-label-old"></span>
                            <span class="neo-label-new neo-filename-label"></span>
                        </label>

                        <label class="neo-rename-dialog-option">
                            <input type="checkbox" checked id="neo-slug-checkbox">
                            <b class="neo-option-name">${neo__("Slug", "Slug")}</b>
                            <span class="neo-label-old neo-slug-label-old"></span>
                            <span class="neo-label-new neo-slug-label"></span>
                        </label>
                    </div>
                </div>
            </div>
            <button class="neo-close-button">&times;</button>
        </div>
        <div class="neo-rename-dialog-warning">
            <p>
                <b>${neo__("Caution:", "Achtung:")}</b>
                ${neo__("Close all page builders and make a backup before renaming.", "Vor dem Umbenennen alle Pagebuilder schließen und Backup erstellen.")}
            </p>
        </div>
    </div>
</div>
`;


function debounce(func, wait) { 
    let timeout; 
    return (...args) => { 
        return new Promise((resolve) => { 
            const later = () => { 
                timeout = null; 
                resolve(func(...args)); 
            };
            clearTimeout(timeout); 
            timeout = setTimeout(later, wait); 
        });
    };
}

async function makeFilenameAndSlugUnique(imgUrl, filename, slug, abortSignal) {
    
    
    if (filename === "") { return ""; }
    const response = await fetch(addQueryParams(neoRenameData.renameUniqueEndpointUrl, { "img-url": imgUrl, "filename": filename, "slug": slug }), { headers: { "X-WP-Nonce": neoRenameData.renameUniqueEndpointNonce }, signal: abortSignal });
    if (!response.ok) { throw new Error("Could not make filename unique"); }
    return extractJson(response);
}
const debouncedMakeFilenameAndSlugUnique = debounce(makeFilenameAndSlugUnique, 200);

function decodeHtmlEntities(str) { 
    const temp = document.createElement('template'); 
    temp.innerHTML = str;
    return temp.content.textContent; 
}

async function getImageInfo(imgUrl) {
    const response = await fetch( 
        addQueryParam(neoRenameData.renameInfoEndpointUrl, "img-url", imgUrl), 
        { method: "GET", headers: { "X-WP-Nonce": neoRenameData.renameInfoEndpointNonce } } 
    );
    if (!response.ok) { let errorMessage = await response.text(); try { errorMessage = JSON.parse(errorMessage).message || errorMessage; } catch {} throw new Error(neo__("Could not get image info!", "Konnte Bildinformationen nicht abrufen!") + " " + errorMessage); } 
    const { filename, slug, title } = await extractJson(response); 
    return { filename: decodeHtmlEntities(filename), slug: decodeHtmlEntities(slug), title: decodeHtmlEntities(title) }; 
}

export async function interfaceOpenRenameDialog20250302(imgUrl) { 
    
    document.activeElement.blur();

    
    const dialogNode = document.createElement("div");
    document.body.appendChild(dialogNode); 
    dialogNode.outerHTML = dialogCode;
    const dialog = document.querySelector(".neo-rename-dialog");

    
    dialog.querySelector(".neo-rename-dialog-backdrop").addEventListener("click", closeDialog);

    const titleInput = dialog.querySelector(".title-input");
    const renameButton = dialog.querySelector(".rename-button");

    function showLoading() {
        titleInput.disabled = true;
        renameButton.disabled = true;
        renameButton.classList.add("loading");
        dialog.querySelector(".neo-rename-dialog-options-container").classList.add("neo-hidden"); 
        dialog.querySelector(".neo-rename-dialog-warning")          .classList.remove("shown"); 
    }
    function hideLoading() {
        titleInput.disabled = false;
        renameButton.disabled = false;
        renameButton.classList.remove("loading");
        dialog.querySelector(".neo-rename-dialog-options-container").classList.remove("neo-hidden"); 
        showOrHideFileRenameWarning(); 
    }

    
    const imagePreview = dialog.querySelector(".image-preview"); 
    const imgUrlCacheBusted = fitProtocolToFetchImgUrl(addCacheBust(imgUrl)); 
    imagePreview.src = imgUrlCacheBusted; 
    imagePreview.addEventListener("error", () => imagePreview.style.display = "none"); 

    
    showLoading();
    let imageInfo; 
    try {
        imageInfo = await getImageInfo(imgUrl); 
    } catch (err) {
        neoError(err);
        alert(err.message);
        closeDialog();
        return;
    }
    let { title, filename, slug } = imageInfo; 
    
    const oldTitle = title;
    const oldFilename = filename;
    const oldSlug = slug;

    
    
    if (title.toLowerCase() === oldFilename.toLowerCase()) {
        title = title.replace(/\.[^.]+$/, "");
    }

    const closeButton = dialog.querySelector(".neo-close-button");
    const titleLabel = dialog.querySelector(".neo-title-label");
    const titleLabelOld = dialog.querySelector(".neo-title-label-old");
    const titleCheckbox = dialog.querySelector("#neo-title-checkbox");
    const filenameLabel = dialog.querySelector(".neo-filename-label");
    const filenameLabelOld = dialog.querySelector(".neo-filename-label-old");
    const filenameCheckbox = dialog.querySelector("#neo-filename-checkbox");
    const slugLabel = dialog.querySelector(".neo-slug-label");
    const slugLabelOld = dialog.querySelector(".neo-slug-label-old");
    const slugCheckbox = dialog.querySelector("#neo-slug-checkbox");

    titleInput.addEventListener("input", updateFields);
    titleCheckbox.addEventListener("change", updateFields);
    filenameCheckbox.addEventListener("change", updateFields);
    slugCheckbox.addEventListener("change", updateFields);

    
    titleInput.value = title;

    
    titleCheckbox.checked = localStorage.getItem("neo-rename-dialog-title-checkbox")       !== "false"; 
    filenameCheckbox.checked = localStorage.getItem("neo-rename-dialog-filename-checkbox") !== "false"; 
    slugCheckbox.checked = localStorage.getItem("neo-rename-dialog-slug-checkbox")         !== "false"; 

    let uniqueFilenameAndSlugAbortController = null;

    
    let emojiList = {}; 
    function replaceEmoji(text) { 
        for (const [emoji, emojiName] of Object.entries(emojiList)) { 
            text = text.replaceAll(emoji, " " + emojiName + " "); 
        }
        return text;
    }
    fetch(neoRenameData.pluginUrl + "/_global-emoji-names.json").then(extractJson).then(emojiListResponse => { 
        emojiList = emojiListResponse.emojis; 
        updateFields(); 
    });

    
    function generateSlug(title) {
        
        let slug = title;
        slug = replaceEmoji(slug); 
        slug = slug.toLowerCase(); 
        slug = slug.replace(/ä/g, "ae").replace(/ö/g, "oe").replace(/ü/g, "ue"); 
        slug = slug.replace(/ä/g, "ae").replace(/ö/g, "oe").replace(/ü/g, "ue"); 
        slug = slug.replace(/ß/g, "ss"); 
        slug = slug.replace(/[^a-z0-9]/g, "-"); 
        slug = slug.replace(/-+/g, "-"); 
        slug = slug.replace(/^-+/, "").replace(/-+$/, ""); 
        return slug;
    }
    function generateFilename(title) {
        const slug = generateSlug(title);
        if (!slug) { return ""; }
        const extension = oldFilename.split(".").pop();
        return slug.replace(/\.[^.]+$/, "") + "." + extension;
    }

    function isDisabled() {
        
        const titleValid = !!titleInput.value;
        if (titleCheckbox.checked && !titleValid) { return true; }
        const filenameValid = !!generateFilename(titleInput.value);
        if (filenameCheckbox.checked && !filenameValid) { return true; }
        const slugValid = !!generateSlug(titleInput.value);
        if (slugCheckbox.checked && !slugValid) { return true; }
        return false;
    }

    
    function applyStrikeThrough(label, strikeThrough) {
        if (strikeThrough) {
            label.classList.add("neo-strike-through");
        } else {
            label.classList.remove("neo-strike-through");
        }
    }
    function showOrHideFileRenameWarning() {
        if (filenameCheckbox?.checked) { 
            dialog.querySelector(".neo-rename-dialog-warning").classList.add("shown"); 
        } else { 
            dialog.querySelector(".neo-rename-dialog-warning").classList.remove("shown"); 
        }
    }
    function updateFields() {
        const title = titleInput.value;
        const filename = generateFilename(titleInput.value);
        const slug = generateSlug(titleInput.value);
        titleLabel.textContent = title;
        filenameLabel.textContent = filename;
        slugLabel.textContent = slug;
        applyStrikeThrough(titleLabel, !titleCheckbox.checked && title !== oldTitle);
        applyStrikeThrough(titleLabelOld, titleCheckbox.checked && title !== oldTitle);
        applyStrikeThrough(filenameLabel, !filenameCheckbox.checked && filename !== oldFilename);
        applyStrikeThrough(filenameLabelOld, filenameCheckbox.checked && filename !== oldFilename);
        applyStrikeThrough(slugLabel, !slugCheckbox.checked && slug !== oldSlug);
        applyStrikeThrough(slugLabelOld, slugCheckbox.checked && slug !== oldSlug);
        showOrHideFileRenameWarning();
        renameButton.disabled = isDisabled();
        uniqueFilenameAndSlugAbortController?.abort("outdated"); 
        if (filenameCheckbox.checked) {
            uniqueFilenameAndSlugAbortController = new AbortController();
            debouncedMakeFilenameAndSlugUnique(imgUrl, generateFilename(titleInput.value), generateSlug(titleInput.value), uniqueFilenameAndSlugAbortController.signal).then((uniqueFilenameAndSlug) => {
                
                filenameLabel.textContent = uniqueFilenameAndSlug.filename;
                applyStrikeThrough(filenameLabel, !filenameCheckbox.checked && uniqueFilenameAndSlug.filename !== oldFilename);
                applyStrikeThrough(filenameLabelOld, filenameCheckbox.checked && uniqueFilenameAndSlug.filename !== oldFilename);
                
                slugLabel.textContent = uniqueFilenameAndSlug.slug;
                applyStrikeThrough(slugLabel, !slugCheckbox.checked && uniqueFilenameAndSlug.slug !== oldSlug);
                applyStrikeThrough(slugLabelOld, slugCheckbox.checked && uniqueFilenameAndSlug.slug !== oldSlug);
            }).catch((err) => err !== "outdated" ? neoError(err) : null); 
        }
        
        titleLabelOld.textContent    = oldTitle;    
        filenameLabelOld.textContent = oldFilename; 
        slugLabelOld.textContent     = oldSlug;     
        
        const tableWidth          = dialog.querySelector(".neo-rename-dialog-options").offsetWidth;           
        const tableContainerWidth = dialog.querySelector(".neo-rename-dialog-options-container").offsetWidth; 
        if (tableWidth > tableContainerWidth) { 
            const char = document.createElement("span"); char.textContent = "n"; dialog.querySelector(".neo-rename-dialog-options").appendChild(char); const charWidth = char.offsetWidth; char.remove(); 
            const currentNumberOfCharacters = Math.max(oldTitle.length, oldFilename.length, oldSlug.length); 
            const maxOldLength = Math.max(10, parseInt(currentNumberOfCharacters * 0.7 - (tableWidth - tableContainerWidth) / charWidth)); 
            if (oldTitle.length    > maxOldLength) { titleLabelOld.textContent    = oldTitle.substring(0, maxOldLength - 1).trimEnd() + "…"; }                                              
            if (oldSlug.length     > maxOldLength) { slugLabelOld.textContent     = oldSlug.substring(0, maxOldLength - 1) + "…"; }                                                         
            if (oldFilename.length > maxOldLength) { filenameLabelOld.textContent = oldFilename.substring(0, maxOldLength - 1 - 9) + "…" + oldFilename.substring(oldFilename.length - 9); } 
        }
        
        localStorage.setItem("neo-rename-dialog-title-checkbox",    titleCheckbox.checked);
        localStorage.setItem("neo-rename-dialog-filename-checkbox", filenameCheckbox.checked);
        localStorage.setItem("neo-rename-dialog-slug-checkbox",     slugCheckbox.checked);
    }

    updateFields(); 
    hideLoading();
    titleInput.focus(); 
    titleInput.select(); 

    let renameResolve, renameReject; 
    const renamedPromise = new Promise((resolve, reject) => { renameResolve = resolve; renameReject = reject; }); 
    renameButton.addEventListener("click", async evt => {
        evt.preventDefault();

        
        const title = titleCheckbox.checked ? titleLabel.textContent : oldTitle;
        const filename = filenameCheckbox.checked ? filenameLabel.textContent : oldFilename;
        const slug = slugCheckbox.checked ? slugLabel.textContent : oldSlug;

        try {
            showLoading();
            const renameResponse = await rename(imgUrl, { title, filename, slug }); 
            hideLoading();
            renameResolve(renameResponse); 
            closeDialog(); 
        } catch (err) {
            neoError(err);
            alert(neo__("Renaming failed!", "Umbenennen fehlgeschlagen!") + " " + err.message);
            hideLoading();
            renameReject(err);
            return;
        }
    });

    function onKeyPress(evt) {
        if (evt.key === "Escape") {
            closeDialog();
        } else if (evt.key === "Enter") {
            renameButton.click();
        }
    }
    window.addEventListener("keydown", onKeyPress);
    
    function closeDialog() {
        dialog.remove();
        window.removeEventListener("keydown", onKeyPress);
    }
    closeButton.addEventListener("click", closeDialog); 

    return renamedPromise; 
}


observeOnce(".neo-media-library-inline-rename-button", button => {
    const imgUrl = button.getAttribute("data-neo-img-url");
    button.addEventListener("click", async (evt) => {
        evt.preventDefault();
        await interfaceOpenRenameDialog20250302(imgUrl);
        reloadPage();
    })
});
